/*
	This is a part of the source code for Pro/DESKTOP.
	Copyright (C) 1998-2002 Parametric Technology Corporation.
	All rights reserved.
*/

#include "stdafx.h"
#include "CISection.h"

//----------------------------------------------------------------------------------------------------------------------------------
// Declarations of Global Variables

IApplication *pdApp;
extern IHelm *pHelm;

//----------------------------------------------------------------------------------------------------------------------------------

HRESULT RunProDESKTOP(IApplication **app) 
{
	// Invokes ProDESKTOP Application and sets the visiblity ON

	START_METHOD("RunProDESKTOP")
	
	pdApp = NULL;

	HRESULT hr = S_OK;

	hr = CoCreateInstance(CLSID_ProDESKTOP, NULL, CLSCTX_SERVER, IID_IApplication, (void**)&pdApp);
	CHECK_RETURN_AND_VAR_STATUS(hr, pdApp)

	hr = (pdApp)->SetVisible(TRUE);
	CHECK_RETURN_STATUS(hr)

	*app = pdApp;

	END_METHOD("RunProDESKTOP")
}

HRESULT NewPart(IPartDocument **retval)
{
	// Create New Part and return the newly created Part.

	START_METHOD("NewPart")

	HRESULT hr = S_OK;
	IApplication *pdApplication = NULL;

	// Get the application
	hr = GetProDESKTOPApplication(&pdApplication);
	CHECK_RETURN_AND_VAR_STATUS(hr, pdApplication)

	IPartDocument *pPartDoc = NULL;
	hr = pdApplication->NewPart(&pPartDoc);
	CHECK_RETURN_AND_VAR_STATUS(hr, pPartDoc)

	*retval = pPartDoc;

	END_METHOD("NewPart")
}

HRESULT CloseProDESKTOP() 
{
	// Closes ProDESKTOP Application and releases the Application pointer.

	START_METHOD("CloseProDESKTOP")

	HRESULT hr = S_OK;

	CHECK_VAR_STATUS(pdApp)

	hr = pdApp->SetVisible(FALSE);
	CHECK_RETURN_STATUS(hr)

	while (hr = pdApp->Release()) {}

	pdApp = NULL;

	END_METHOD("CloseProDESKTOP")
}

HRESULT GetProDESKTOPApplication(IApplication **ProDESKTOPApp)
{
	// Returns the ProDESKTOP Application

	START_METHOD("GetProDESKTOPApplication")

	*ProDESKTOPApp = NULL;
	
	CHECK_VAR_STATUS(pdApp)

	*ProDESKTOPApp = pdApp;

	END_METHOD("GetProDESKTOPApplication")
}

HRESULT GetActivePartWorkplaneSketch(IPartDocument **activePart,
									 IGraphicDocument **activeGraphicDoc,
									 IWorkplane **activeWorkplane,
									 ISketch **activeSketch) 
{
	// Queries the  PartDocument, Workplane , and Sketch Interfaces and
	// returns the active Part, active Workplane, active Sketch

	START_METHOD("GetActivePartWorkplaneSketch")

	HRESULT hr = S_OK;

	IPartDocument *part = NULL;
	IWorkplane *workplane = NULL;

	ISketch * sketch = NULL;
	IApplication *pdApplication = NULL;

	// Get the application
	hr = GetProDESKTOPApplication(&pdApplication) ;
	CHECK_RETURN_AND_VAR_STATUS(hr, pdApplication)

	hr = pdApplication->GetActiveDoc(activeGraphicDoc);
	CHECK_RETURN_AND_VAR_STATUS(hr, activeGraphicDoc)

	hr = (*activeGraphicDoc)->QueryInterface(IID_IPartDocument, (void**)activePart);
	CHECK_RETURN_AND_VAR_STATUS(hr, activePart)

	(*activeGraphicDoc)->Release();

	// Get active workplane
	hr = (*activeGraphicDoc)->GetActiveWorkplane(activeWorkplane);
	CHECK_RETURN_AND_VAR_STATUS(hr, activeWorkplane)

	// Get active sketch
	hr = (*activeGraphicDoc)->GetActiveSketch(activeSketch);
	CHECK_RETURN_AND_VAR_STATUS(hr, activeSketch)

	END_METHOD("GetActivePartWorkplaneSketch")
}

HRESULT CreateLine(double startX, double startY, double endX, double endY, ILine **dCurve1)
{
	// Creates a line using the two given points and returns the line created
	// after creation the line is AutoConstrained.

	START_METHOD("CreateLine")

	HRESULT hr = S_OK;
	
	IApplication *app = NULL;

	hr = GetProDESKTOPApplication(&app);
	CHECK_RETURN_AND_VAR_STATUS(hr, app)

	IGraphicDocument *activeGraphicDoc;
	IPartDocument * activePart = NULL;
	IWorkplane * activeWorkplane = NULL;
	ISketch * activeSketch = NULL;

	hr = GetActivePartWorkplaneSketch(&activePart, &activeGraphicDoc, &activeWorkplane, &activeSketch);
	if (FAILED(hr) || !activePart || !activeWorkplane || !activeSketch)
		return E_FAIL;

	// Create the startVector
	IVector *startVector2D = NULL;

	hr = (GetCLASS(Vector))->CreateVector(startX, startY, 0, &startVector2D);
	CHECK_RETURN_AND_VAR_STATUS(hr, startVector2D)

	// Get the 3D vector for startVector
	IVector *startVector3D = NULL;
	hr = activeWorkplane->Get3DVector(startVector2D, &startVector3D);
	CHECK_RETURN_AND_VAR_STATUS(hr, startVector3D)

	// Create the endVector
	IVector *endVector2D = NULL;
	hr = (GetCLASS(Vector))->CreateVector(endX, endY, 0, &endVector2D);
	CHECK_RETURN_AND_VAR_STATUS(hr, startVector2D)

	// Get the 3D vector for endVector
	IVector *endVector3D = NULL;
	hr = activeWorkplane->Get3DVector(endVector2D, &endVector3D);
	CHECK_RETURN_AND_VAR_STATUS(hr, startVector3D)

	// Create basic straight line
	IBasicStraight *curve1 = NULL;

	// Currently, we are using the startVector2D and endVector2D
	// which are in the active workplane local co-ordinates.
	// These will be changed to take "Global Co-ordinates - 3D vectors
	// startVector3D, endVector3D

	hr = (GetCLASS(BasicStraight))->CreateBasicStraightTwoPoints(startVector3D, endVector3D, &curve1);
	CHECK_RETURN_AND_VAR_STATUS(hr, curve1)

	// Create aLine object
	ILine *line1 = NULL;
	hr = activeSketch->CreateLine(CAST(ICurve, curve1), &line1);
	CHECK_RETURN_AND_VAR_STATUS(hr, line1)

	*dCurve1 = line1;

	// Apply AutoConstrain on the line created
	hr = activeWorkplane->AutoConstrain(CAST(IObjectOrSet, line1));
	CHECK_RETURN_AND_VAR_STATUS(hr, line1)

	// Set the selection to the line created
	hr = activeGraphicDoc->SetSelection(CAST(IObjectOrSet, line1));
	CHECK_RETURN_STATUS(hr)

	// Update the views
	CString str = "Operation";
	hr = pHelm->CommitCalls(str.AllocSysString(), FALSE);

	CHECK_RETURN_STATUS(hr)

	END_METHOD("CreateLine")
}

HRESULT CreateExtrusion(ISketch *pSketch, double distanceAbove, int distanceBelow, double taperAngle, int side, long materialStatus, CString extrusionName, IExtrusion **pExtrusion)
{
	// Performs extrusion on Active Design and return the Extrusion it created.

	START_METHOD("CreateExtrusion")

	HRESULT hr = S_OK;

	IApplication *pdApplication = NULL;
	IGraphicDocument *activeGraphicDoc = NULL;
	IPartDocument *activePart = NULL;
	IWorkplane *activeWorkplane = NULL;
	ISketch *activeSketch = NULL;

	// Get the application
	hr = GetProDESKTOPApplication(&pdApplication);
	CHECK_RETURN_AND_VAR_STATUS(hr, pdApplication)

	hr = GetActivePartWorkplaneSketch(&activePart, &activeGraphicDoc, &activeWorkplane, &activeSketch);
	if (FAILED(hr) || !(activeGraphicDoc) || !(activePart) || !(activeWorkplane) || !(activeSketch)) {
		AfxMessageBox("Could not get active part/workplane/sketch");
		return E_FAIL;
	}

	// Getting the Helm
	IDispatch *pDisp = NULL;

	IDesign *pDesign = NULL;
	hr = activePart->GetDesign(&pDesign);
	CHECK_RETURN_AND_VAR_STATUS(hr, pDesign)

	IExtrusion *extrusion = NULL;
	
	hr = (GetCLASS(Extrusion))->CreateExtrusion(pDesign, pSketch, distanceAbove, distanceBelow, taperAngle, side, materialStatus, 0, &extrusion);
	CHECK_RETURN_AND_VAR_STATUS(hr, extrusion)

	IOperation *operation = NULL;
	hr = extrusion->QueryInterface(IID_IOperation, (void **)&operation);
	CHECK_RETURN_AND_VAR_STATUS(hr, operation)

	hr = operation->SetName(extrusionName.AllocSysString());
	CHECK_RETURN_STATUS(hr)

	*pExtrusion = extrusion;

	// Committing the operation
	CString op = "Operation";
	hr = pHelm->CommitCalls(op.AllocSysString(), FALSE);

	CHECK_RETURN_STATUS(hr)

	END_METHOD("CreateExtrusion")

}

HRESULT ConstantRadiusBlending(ISet *pInputEdgeSet, VARIANT_BOOL bChamfer, VARIANT_BOOL bFlipped, VARIANT_BOOL bPropagate, VARIANT_BOOL bSmoothVariation, VARIANT_BOOL bGroupEdges, double radius, CString blendName, IBlend **pBlend) 
{
	// Blends with the constant radius and return the newly created blend.
	// if Chamfer is set then it is constant Chamfering.Other paramrters are
	// self explanatory.

	START_METHOD("ConstantRadiusBlending")

	HRESULT  hr = S_OK;
	IApplication *pdApplication = NULL;
	IGraphicDocument *activeGraphicDoc = NULL;
	IPartDocument *activePart = NULL;
	IWorkplane *activeWorkplane = NULL;
	ISketch *activeSketch = NULL;

	// Get the application
	hr = GetProDESKTOPApplication(&pdApplication);
	CHECK_RETURN_AND_VAR_STATUS(hr, pdApplication)

	hr = GetActivePartWorkplaneSketch(&activePart, &activeGraphicDoc, &activeWorkplane, &activeSketch);
	if (FAILED(hr) || !activePart || !activeWorkplane || !activeSketch) {
		AfxMessageBox("Could not get active part/workplane/sketch");
		return E_FAIL;
	}

	IDesign *pDesign = NULL;
	hr = activePart->GetDesign(&pDesign);
	CHECK_RETURN_AND_VAR_STATUS(hr, pDesign)

	IBlend *blend = NULL;
	hr = (GetCLASS(Blend))->CreateBlend(pDesign, CAST(IObjectOrSet, pInputEdgeSet), bChamfer, 0, radius ,0, &blend);
	CHECK_RETURN_AND_VAR_STATUS(hr, blend)

	IOperation *operation=NULL ;
	hr = blend->QueryInterface(IID_IOperation, (void **)&operation);
	CHECK_RETURN_AND_VAR_STATUS(hr, operation)

	hr = operation->SetName(blendName.AllocSysString()) ;
	CHECK_RETURN_STATUS(hr)

	*pBlend = blend ;

	hr = activePart->UpdateDesign() ;
	CHECK_RETURN_STATUS(hr)

	// Committing the operation
	CString op = "Operation";
	hr = pHelm->CommitCalls(op.AllocSysString(), FALSE);

	CHECK_RETURN_STATUS(hr)

	END_METHOD("ConstantRadiusBlending")
}

HRESULT CreateISection(double length, double verticalThickness,
					   double horizontalThickness, double width,
					   double extrusionHeight, double blendRadius, ISet **dObjectSet1)
{
	// Creates an ISection sketch and then extrudes it based on the parameters given

	START_METHOD("CreateISection")

	HRESULT hr = NOERROR;

	IGraphicDocument *activeGraphicDoc = NULL;
	IPartDocument * activePart = NULL;
	IWorkplane * activeWorkplane = NULL;
	ISketch * activeSketch = NULL;

	hr = GetActivePartWorkplaneSketch(&activePart, &activeGraphicDoc, &activeWorkplane, &activeSketch);
	if (FAILED(hr) || !activePart || !activeWorkplane || !activeSketch) {
		AfxMessageBox("Could not get active part/workplane/sketch");
		return E_FAIL;
	}

	double k = (width - verticalThickness)/2.0 ;

	ILine *line1 = NULL;
	hr = CreateLine(0, 0, 0, horizontalThickness, &line1);
	CHECK_RETURN_AND_VAR_STATUS(hr, line1)

	ILine *line2 = NULL;
	hr = CreateLine(0, horizontalThickness, k, horizontalThickness, &line2);
	CHECK_RETURN_AND_VAR_STATUS(hr, line2)

	ILine *line3 = NULL;
	hr = CreateLine(k, horizontalThickness, k, length + horizontalThickness , &line3);
	CHECK_RETURN_AND_VAR_STATUS(hr, line3)

	ILine *line4 = NULL;
	hr = CreateLine(k, length + horizontalThickness, 0, length + horizontalThickness , &line4);
	CHECK_RETURN_AND_VAR_STATUS(hr, line4)

	ILine *line5 = NULL;
	hr = CreateLine(0, length + horizontalThickness, 0, length + 2 * horizontalThickness , &line5);
	CHECK_RETURN_AND_VAR_STATUS(hr, line5)

	ILine *line6 = NULL;
	hr = CreateLine(0, length + 2 * horizontalThickness, width, length + 2 * horizontalThickness , &line6);
	CHECK_RETURN_AND_VAR_STATUS(hr, line6)

	ILine *line7 = NULL;
	hr = CreateLine(width, length + 2 * horizontalThickness, width, length + horizontalThickness , &line7);
	CHECK_RETURN_AND_VAR_STATUS(hr, line7)

	ILine *line8 = NULL;
	hr = CreateLine(width, length + horizontalThickness, k + verticalThickness, length + horizontalThickness , &line8);
	CHECK_RETURN_AND_VAR_STATUS(hr, line8)

	ILine *line9 = NULL;
	hr = CreateLine(k + verticalThickness, length + horizontalThickness, k + verticalThickness, horizontalThickness , &line9);
	CHECK_RETURN_AND_VAR_STATUS(hr, line9)

	ILine *line10 = NULL;
	hr = CreateLine(k + verticalThickness, horizontalThickness, width, horizontalThickness , &line10);
	CHECK_RETURN_AND_VAR_STATUS(hr, line10)

	ILine *line11 = NULL;
	hr = CreateLine(width, horizontalThickness, width, 0 , &line11);
	CHECK_RETURN_AND_VAR_STATUS(hr, line11)

	ILine *line12 = NULL;
	hr = CreateLine(width, 0, 0, 0 , &line12);
	CHECK_RETURN_AND_VAR_STATUS(hr, line12)

	// Create a set of lines
	ISet *aLineSet1 = NULL;
	(GetCLASS(ObjectSet))->CreateAObjectSet(&aLineSet1);
	CHECK_RETURN_AND_VAR_STATUS(hr, aLineSet1)

	// Add all the sides of the rectangle to the set
	aLineSet1->AddMember(line1);
	aLineSet1->AddMember(line2);

	aLineSet1->AddMember(line3);
	aLineSet1->AddMember(line4);

	aLineSet1->AddMember(line5);
	aLineSet1->AddMember(line6);

	aLineSet1->AddMember(line7);
	aLineSet1->AddMember(line8);

	aLineSet1->AddMember(line9);
	aLineSet1->AddMember(line10);

	aLineSet1->AddMember(line11);
	aLineSet1->AddMember(line12);

	// Apply AutoConstrain on the set of lines. AutoConstrain is called here
	// only once as an optimization, after all the lines are created.
	activeWorkplane->AutoConstrain(CAST(IObjectOrSet, aLineSet1));
	*dObjectSet1 = aLineSet1;

	// Set the selection to the object set
	hr = activeGraphicDoc->SetSelection(CAST(IObjectOrSet, aLineSet1));
	CHECK_RETURN_STATUS(hr)

	IExtrusion *pExtrusion = NULL;
	
	/* The meaning of the fifth variable 'side' has changed from version V6 onwards
	a value of 0 means it is inside and val of 1 means it is outer */
	
	hr = CreateExtrusion(activeSketch, extrusionHeight,0, 0, 0, 1,"extrusion", &pExtrusion);
	CHECK_RETURN_AND_VAR_STATUS(hr, pExtrusion)

	hr = activePart->UpdateDesign();
	CHECK_RETURN_STATUS(hr)

	// Update the views
	IPartOthers *part1 = NULL;
	hr = activePart->QueryInterface(IID_IPartOthers, (void **)&part1);
	part1->Release();
	CHECK_RETURN_STATUS(hr)

	IDesign *pDesign = NULL;
	ISet *pEdgeSet = NULL;

	IBlend *pBlend = NULL;
	IDesign *localDesign = NULL;

	if (blendRadius > 0) {
		hr = activePart->GetDesign(&pDesign);
		CHECK_RETURN_AND_VAR_STATUS(hr, pDesign)

		hr = pDesign->QueryInterface(IID_IDesign, (void **)&localDesign);
		CHECK_RETURN_AND_VAR_STATUS(hr, localDesign)

		hr = localDesign->GetEdges(&pEdgeSet);
		CHECK_RETURN_AND_VAR_STATUS(hr, pEdgeSet)

		hr = ConstantRadiusBlending(pEdgeSet, FALSE, FALSE, FALSE, FALSE, FALSE, blendRadius, "const blend", &pBlend) ;
		CHECK_RETURN_AND_VAR_STATUS(hr, pBlend)

		hr = activePart->UpdateDesign() ;
		CHECK_RETURN_STATUS(hr)
	}
	// Committing the operation
	CString op = "Operation";
	hr = pHelm->CommitCalls(op.AllocSysString(), FALSE);

	END_METHOD("CreateISection")
}
